﻿/*	VERSION: 1.2


USAGE:
	Create a "dragHandler" movieClip in the library.
	Attach this class to it.
	Place the "dragHandler" movieClip on the stage, above everything else (or highest depth)
	For each movieClip/component that will send drag actions:
		drag = _parent.dragHandler_mc.makeDraggable( this );
		drag.onSendClick = function( dragHandler_mc ){
			return {data:dragData, image:image, imageOffset:imageOffset};
		);
		drag.onSendDrag = function(){
			return {data:dragData, image:image, imageOffset:imageOffset};
		);
		drag.onSendDrop = function( dragData, dropTarget, isValid ){);
		drag.onRecieveDrop = function( dragData ){);
	
	
	Drop-targets only need this:
		drag = _parent.dragHandler_mc.makeDraggable( this );
		drag.onRecieveDrop = function( dragData ){);
	
	
	
NOTES:
	Both sending and recieving movieClips need their own drag objects,
	otherwise, they cannot communicate.
	Drag objects are created using the makeDraggable() function.
	
	MovieClips can recieve drops from themselves.
	
	MovieClips can send data to undefined
	
	Irrelevent areas of movieClips and components should explicitly send empty data
	by clearing their dragData.data variable.
	(usually within their onSendClick() or onSendDrag() functions)
	
	Be prepared to recieve undefined data from other movieClips.
	Typically, this will occur when the user clicks a drop-target and drags to a movieClip that has drag-&-drop.
	
	
	
CONTENTS OF dragData:
	parent:movieClip,			// hitTest click-target for this set of drag events
	data:{},							// arbitrary data to send to the drop target
	isClicked:false,			// whether parent movieClip is being pressed
	isDragging:false,			// whether this drag is active
	clickPoint:Point,			// where the clicked occured (global-relative)
	dragPoint:Point,			// where the drag started occuring  (global-relative)
	dropPoint:Point,			// where the drop occured (global-relative)
	onSendClick:Function,	// reaction to clicking on the parent
	onSendDrag:Function,	// reaction when the drag begins
	onRecieveDrop:Function,		// reaction to recieving a drop from another movieClip
	onSendDrop:Function		// reaction when the drop occurs
	
	The functions are defined externally, as demonstrated in the usage example above.
	The return data sent by onSendClick and onSendDrag are all optional.
*/



class classes.DragAndDrop extends MovieClip
{
	// _______________________________________
	// variables
	var _this:MovieClip;
	var preview:MovieClip;
	var imageOffset:flash.geom.Point;
	var dragList = [];
	var mouse:Object = {};
	var dragThreshold = 8;
	// _______________________________________
	// init
	function DragAndDrop(){
		_visible = false;
		_this = this;
		trace("_this: "+_this);
		preview = this.createEmptyMovieClip("preview", 0);
		imageOffset = new flash.geom.Point(0,0);
		setupMouse();
	}// DragAndDrop()
	// destructor
	function onUnload(){
		Mouse.removeListener( mouse );
	}
	function checkCleanUp(){
		if(_name == undefined)
			onUnload();
	}// checkCleanUp()
	// _______________________________________
	// properties
	var _image:flash.display.BitmapData = null;
	function get image(){
		return _image;
	}
	function set image( newBitmap:flash.display.BitmapData ){
		if(newBitmap == null)
			var newBitmap = new flash.display.BitmapData( 1,1 );
		preview.attachBitmap( newBitmap, 0 );
		_image = newBitmap;
		preview._x = -_this.imageOffset.x;
		preview._y = -_this.imageOffset.y;
	}
	// _______________________________________
	// main programming
	function makeDraggable( movieClip:MovieClip )
	{
		var newDrag = {
			parent:movieClip,			// hitTest click-target for this set of drag events
			data:{},							// arbitrary data to send to the drop target
			isClicked:false,			// whether parent movieClip is being pressed
			isDragging:false,			// whether this drag is active
			clickPoint:new flash.geom.Point(0,0),			// where the clicked occured (global-relative)
			dragPoint:new flash.geom.Point(0,0),			// where the drag started occuring  (global-relative)
			dropPoint:new flash.geom.Point(0,0),			// where the drop occured (global-relative)
			onSendClick:Function,	// reaction to clicking on the parent
			onSendDrag:Function,	// reaction when the drag begins
			onRecieveDrop:Function,		// reaction to recieving a drop from another movieClip
			onSendDrop:Function		// reaction when the drop occurs
		}
		dragList.push( newDrag );
		return newDrag;
	}// makeDraggable()
	function removeDrag( movieClip )
	{
		for(var i in dragList)
			if(dragList[i].parent == movieClip)
				dragList.splice( i, 1 );
	}// removeDrag()
	
	
	
	// __________________________________________________
	function setupMouse()
	{
		var _this = this._this;		// allow cross-scope access to the class object
		mouse = {
			_this:_this,
			onMouseDown:function()
			{
				_this.checkCleanUp();
				// find the movieClip that's been clicked on 
				for(var i=0; i<_this.dragList.length; i++)
				{// for:  each drag object
					var drag_obj = _this.dragList[i];
					if( drag_obj.parent.hitTest(_root._xmouse, _root._ymouse, true) )
					{// if:  this movieClip was clicked
						// remember that this movieClip was clicked on
						drag_obj.isClicked = true;
						// remember the click coordinates (global-relative)
						drag_obj.clickPoint = new flash.geom.Point(drag_obj.parent._xmouse,drag_obj.parent._ymouse);
						drag_obj.parent.localToGlobal( drag_obj.clickPoint );
						// tell the sender to prepare data for sending
						_this.image = null;
						var newData = drag_obj.onSendClick( _this );
						// add data
						_this.imageOffset = newData.imageOffset || new flash.geom.Point(0,0);
						for(var nam in newData.data)
							drag_obj.data[nam] = newData[nam];
						if(newData.image!=undefined)
							_this.image = newData.image;
					}// if:  this movieClip was clicked
				}// for:  each drag object
			},// click
			
			
			onMouseMove:function()
			{
				_this.checkCleanUp();
				// find the movieClip that was clicked
				for(var i=0; i<_this.dragList.length; i++)
				{// for:  each drag object
					var drag_obj = _this.dragList[i];
					if( drag_obj.isClicked==true )
					{// if: this movieClip was clicked on
						if(drag_obj.isDragging==false)
						{// if: clicked movieClip hasn't started dragging yet
							// remember where this movieClip started being dragged
							drag_obj.dragPoint = new flash.geom.Point(drag_obj.parent._xmouse,drag_obj.parent._ymouse);
							drag_obj.parent.localToGlobal( drag_obj.dragPoint );
							// check whether it's been dragged far enough
							var dragDistance = flash.geom.Point.distance( drag_obj.clickPoint, drag_obj.dragPoint );
							if(dragDistance > _this.dragThreshold)
							{// if:  the movieClip has been dragged far enough to count as deliberate dragging
								drag_obj.isDragging = true;
								// tell the sender that a drag has begun
								var newData = drag_obj.onSendDrag( _this );
								// add data
								_this.imageOffset = newData.imageOffset || new flash.geom.Point(0,0);
								for(var nam in newData.data)
									drag_obj.data[nam] = newData.data[nam];
								if(newData.image!=undefined)
									_this.image = newData.image;
								// appear and start following the mouse
								_this._x = _this._parent._xmouse;
								_this._y = _this._parent._ymouse;
								_this._visible = true;
							}// if:  the movieClip has been dragged far enough to count as deliberate dragging
						}// if: clicked movieClip hasn't started dragging yet
						
						if(drag_obj.isDragging==true)
						{// if: clicked movieClip is being dragged
							// follow the mouse
							_this._x = _this._parent._xmouse;
							_this._y = _this._parent._ymouse;
						}// if: clicked movieClip is being dragged
					}// if: this movieClip was clicked on
				}// for:  each drag object
			},// drag
			
			
			onMouseUp:function()
			{
				_this.checkCleanUp();
				for(var a=0; a<_this.dragList.length; a++)
				{// for:  each drag object
					var drag_obj = _this.dragList[a];
					if(drag_obj.isDragging == true)
					{// if: this object is being dragged
						var isValid = false;
						var drop_obj = undefined;
						var foundDropTarget = false;
						// hitTest against registered movieclips
						for(var b=0; b<_this.dragList.length; b++)
						{// for:  each drag object
							drop_obj = _this.dragList[b];
							if(drop_obj.parent.hitTest(_root._xmouse, _root._ymouse, true) )
							{// if:  dropped on this object
								// tell the target that it's recieving something
								isValid = drop_obj.onRecieveDrop( drag_obj );
								foundDropTarget = true;
							}// if:  dropped on this object
							// disappear
							_this._visible = false;
							// stop dragging
							drag_obj.isClicked = false;
							drag_obj.isDragging = false;
							// only affect one drop-target
							if(foundDropTarget)
								break;
						}// for:  each drag object
						// detect / invalid empty drop-targets
						if(foundDropTarget == false){
							drop_obj = null;
							_this.image = null;
						}
						// tell the sender that the data has been sent, what recieved it, and whether it was accepted
						drag_obj.onSendDrop( drag_obj, drop_obj.parent, isValid );
					}// if: this object is being dragged
				}// for:  each drag object
			}// drop
		}// mouse listener object
		Mouse.addListener( mouse );
	}// setupMouse()
	
}// DragAndDrop class